<?php
    declare(strict_types=1);

    require_once APP_PATH . "/repositories/PreguntaRepository.php";

    class PreguntaService {
        private PreguntaRepository $repo;

        public function __construct() {
            $this->repo = new PreguntaRepository();
        }

        public function listar(int $idDocente): array {
            return $this->repo->getAllByDocente($idDocente);
        }

        public function listarPorCurso(int $idDocente, int $idCurso): array
        {
         return $this->repo->getAllByDocenteCurso($idDocente, $idCurso);
        }



        public function crear(int $idDocente, array $data, ?string $imagen): void {

            $enunciado = trim($data['enunciado'] ?? '');
            $dificultad = trim($data['dificultad'] ?? 'FACIL');
            $grado = strtoupper(trim($data['grado'] ?? 'TODOS'));

            $alt = $data['alt'] ?? [];              // alt[0..3]
            $correctaIndex = $data['correcta'] ?? null;

            if ($enunciado === '') throw new Exception("El enunciado es obligatorio.");
            if (!in_array($dificultad, ['FACIL','MEDIA','DIFICIL'], true)) throw new Exception("Dificultad inválida.");
            if (!in_array($grado, ['1','2','3','4','5','TODOS'], true)) throw new Exception("Grado inválido.");
            if ($correctaIndex === null || $correctaIndex === '') throw new Exception("Selecciona la alternativa correcta.");

            $correctaIndex = (int)$correctaIndex;
            if ($correctaIndex < 0 || $correctaIndex > 3) throw new Exception("Alternativa correcta inválida.");

            // contar cuántas alternativas NO vacías hay
            $countNoVacias = 0;
            for ($i=0; $i<4; $i++) {
                $txt = trim((string)($alt[$i] ?? ''));
                if ($txt !== '') $countNoVacias++;
            }
            if ($countNoVacias < 2) throw new Exception("Ingresa mínimo 2 alternativas.");

            // la alternativa marcada como correcta NO puede estar vacía
            $txtCorrecta = trim((string)($alt[$correctaIndex] ?? ''));
            if ($txtCorrecta === '') throw new Exception("La alternativa marcada como correcta no puede estar vacía.");

            $pdo = db();
            $pdo->beginTransaction();
            try {
                $idCurso = (int)($data['id_curso'] ?? 0);
                $idCurso = $idCurso > 0 ? $idCurso : null;

                $idPregunta = $this->repo->createPregunta($enunciado, $dificultad, $idDocente, $grado, $idCurso, $imagen);


                // Insertar alternativas en orden A..D (solo si no están vacías)
                for ($i=0; $i<4; $i++) {
                $texto = trim((string)($alt[$i] ?? ''));
                if ($texto === '') continue;
                $this->repo->addAlternativa($idPregunta, $texto, $i === $correctaIndex);
                }

                $pdo->commit();
            } catch (Throwable $e) {
                $pdo->rollBack();
                throw $e;
            }
        }
        public function obtener(int $idDocente, int $idPregunta): array {
            $p = $this->repo->findById($idPregunta, $idDocente);
            if (!$p) throw new Exception("Pregunta no encontrada.");
            $alts = $this->repo->getAlternativas($idPregunta);
            return ["pregunta" => $p, "alternativas" => $alts];
            }

        public function actualizar(int $idDocente, int $idPregunta, array $data, ?string $imagen): void {
            $enunciado = trim($data['enunciado'] ?? '');
            $dificultad = trim($data['dificultad'] ?? 'FACIL');
            $grado = strtoupper(trim($data['grado'] ?? 'TODOS'));
            $activo = isset($data['activo']) ? 1 : 0;

            $alt = $data['alt'] ?? [];              // alt[0..3]
            $correctaIndex = $data['correcta'] ?? null;

            if ($enunciado === '') throw new Exception("El enunciado es obligatorio.");
            if (!in_array($dificultad, ['FACIL','MEDIA','DIFICIL'], true)) throw new Exception("Dificultad inválida.");
            if (!in_array($grado, ['1','2','3','4','5','TODOS'], true)) throw new Exception("Grado inválido.");
            if ($correctaIndex === null || $correctaIndex === '') throw new Exception("Selecciona la alternativa correcta.");

            $correctaIndex = (int)$correctaIndex;
            if ($correctaIndex < 0 || $correctaIndex > 3) throw new Exception("Alternativa correcta inválida.");

            $countNoVacias = 0;
            for ($i=0; $i<4; $i++) {
                $txt = trim((string)($alt[$i] ?? ''));
                if ($txt !== '') $countNoVacias++;
            }
            if ($countNoVacias < 2) throw new Exception("Ingresa mínimo 2 alternativas.");

            $txtCorrecta = trim((string)($alt[$correctaIndex] ?? ''));
            if ($txtCorrecta === '') throw new Exception("La alternativa marcada como correcta no puede estar vacía.");

            // valida propiedad
            $p = $this->repo->findById($idPregunta, $idDocente);
            if (!$p) throw new Exception("Pregunta no encontrada.");

            $pdo = db();
            $pdo->beginTransaction();
            try {
                $idCurso = (int)($data['id_curso'] ?? 0);
                $idCurso = $idCurso > 0 ? $idCurso : null;

                $this->repo->updatePregunta($idPregunta, $idDocente, $enunciado, $dificultad, $grado, $activo, $idCurso, $imagen);


                // Reemplazar alternativas
                $this->repo->deleteAlternativas($idPregunta);
                for ($i=0; $i<4; $i++) {
                $texto = trim((string)($alt[$i] ?? ''));
                if ($texto === '') continue;
                $this->repo->addAlternativa($idPregunta, $texto, $i === $correctaIndex);
                }

                $pdo->commit();
            } catch (Throwable $e) {
                $pdo->rollBack();
                throw $e;
            }
        }

        public function eliminar(int $idDocente, int $idPregunta): void {
            $p = $this->repo->findById($idPregunta, $idDocente);
            if (!$p) throw new Exception("Pregunta no encontrada.");

            // Si ya hay respuestas vinculadas, la FK puede impedir borrar.
            // En ese caso, conviene "desactivar" en vez de borrar.
            try {
                $this->repo->deletePregunta($idPregunta, $idDocente);
            } catch (Throwable $e) {
                // fallback: desactivar
                $this->repo->updatePregunta($idPregunta, $idDocente, $p['enunciado'], $p['dificultad'], (string)($p['grado'] ?? 'TODOS'), 0);
                throw new Exception("No se pudo eliminar (probablemente ya fue usada). Se desactivó la pregunta.");
            }
        }


    }
