<?php
declare(strict_types=1);

require_once APP_PATH . "/repositories/PartidaRepository.php";
require_once APP_PATH . "/repositories/PreguntaRepository.php";

class PartidaService {
  private PartidaRepository $repo;
  private PreguntaRepository $pregRepo;

  public function __construct() {
    $this->repo = new PartidaRepository();
    $this->pregRepo = new PreguntaRepository();
  }

  // ✅ Solo para EDITAR/ELIMINAR (estado CREADA)
  public function obtener(int $idDocente, int $idPartida): array {
    // ✅ auto-finalizar si venció (por si quedó colgado)
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");
    if (($p['estado'] ?? '') !== 'CREADA') {
      throw new Exception("Solo se puede editar/eliminar si está CREADA.");
    }
    return $p;
  }

  public function actualizar(int $idDocente, int $idPartida, array $data): void {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");
    if (($p['estado'] ?? '') !== 'CREADA') throw new Exception("Solo se puede actualizar si está CREADA.");

    $titulo = trim((string)($data['titulo'] ?? ''));
    $duracion = (int)($data['duracion_total_min'] ?? 30);
    $gradoObjetivo = (int)($data['grado_objetivo'] ?? 0);
    $idCurso = isset($data['id_curso']) ? (int)$data['id_curso'] : null;
    $modoPreg = strtoupper(trim((string)($data['modo_preguntas'] ?? 'ORDEN')));
    $modoAlt  = strtoupper(trim((string)($data['modo_alternativas'] ?? 'FIJO')));

    if ($titulo === '') throw new Exception("El título es obligatorio.");
    if ($duracion < 1 || $duracion > 300) throw new Exception("Duración total debe estar entre 1 y 300 minutos.");
    if ($idCurso !== null && $idCurso <= 0) $idCurso = null;
    if ($gradoObjetivo < 0 || $gradoObjetivo > 5) $gradoObjetivo = 0;
    if (!in_array($modoPreg, ['ORDEN','ALEATORIO'], true)) $modoPreg = 'ORDEN';
    if (!in_array($modoAlt,  ['FIJO','ALEATORIO'], true)) $modoAlt = 'FIJO';

    $this->repo->updatePartida($idPartida, $idDocente, $titulo, $duracion, $gradoObjetivo, $idCurso, $modoPreg, $modoAlt);
  }

  public function eliminar(int $idDocente, int $idPartida): void {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");
    if (($p['estado'] ?? '') !== 'CREADA') throw new Exception("Solo se puede eliminar si está CREADA.");

    $this->repo->deletePartida($idPartida, $idDocente);
  }

  // ✅ Wrapper (por compatibilidad)
  public function listar(int $idDocente, array $q = []): array {
    return $this->listarFiltrado($idDocente, $q);
  }

  public function listarFiltrado(int $idDocente, array $q): array {
    // ✅ auto-finaliza todas las vencidas del docente
    $this->repo->autoFinalizarVencidasPorDocente($idDocente);

    // CURSO
    $idCurso = null;
    if (isset($q['curso']) && $q['curso'] !== '') {
      $tmp = (int)$q['curso'];
      if ($tmp > 0) $idCurso = $tmp;
    }

    // ESTADO
    $estado = null;
    if (isset($q['estado']) && $q['estado'] !== '') {
      $tmp = strtoupper(trim((string)$q['estado']));
      if ($tmp !== 'TODOS') $estado = $tmp;
    }

    // GRADO (si viene "" => null)
    $grado = null;
    if (isset($q['grado']) && $q['grado'] !== '') {
      $tmp = (int)$q['grado'];
      if ($tmp >= 0 && $tmp <= 5) $grado = $tmp;
    }

    // FECHAS (YYYY-MM-DD)
    $desde = (isset($q['desde']) && $q['desde'] !== '') ? (string)$q['desde'] : null;
    $hasta = (isset($q['hasta']) && $q['hasta'] !== '') ? (string)$q['hasta'] : null;

    return $this->repo->getAllByDocenteFiltrado(
      $idDocente,
      $idCurso,
      $estado,
      $grado,
      $desde,
      $hasta
    );
  }

  private function generarCodigoUnico(): string {
    for ($i=0; $i<10; $i++) {
      $code = "SEIR-" . str_pad((string)random_int(0, 99999), 5, "0", STR_PAD_LEFT);
      if (!$this->repo->codigoExiste($code)) return $code;
    }
    throw new Exception("No se pudo generar código, intenta de nuevo.");
  }

  public function crear(int $idDocente, array $data): void {
    $titulo = trim((string)($data['titulo'] ?? ''));
    $duracion = (int)($data['duracion_total_min'] ?? 30);
    $gradoObjetivo = (int)($data['grado_objetivo'] ?? 0);

    $idCurso = isset($data['id_curso']) ? (int)$data['id_curso'] : null;
    $modoPreg = strtoupper(trim((string)($data['modo_preguntas'] ?? 'ORDEN')));
    $modoAlt  = strtoupper(trim((string)($data['modo_alternativas'] ?? 'FIJO')));

    if ($titulo === '') throw new Exception("El título es obligatorio.");
    if ($duracion < 1 || $duracion > 300) throw new Exception("Duración total debe estar entre 1 y 300 minutos.");
    if ($idCurso !== null && $idCurso <= 0) $idCurso = null;
    if ($gradoObjetivo < 0 || $gradoObjetivo > 5) $gradoObjetivo = 0;

    if (!in_array($modoPreg, ['ORDEN','ALEATORIO'], true)) $modoPreg = 'ORDEN';
    if (!in_array($modoAlt,  ['FIJO','ALEATORIO'], true)) $modoAlt = 'FIJO';

    $codigo = $this->generarCodigoUnico();
    $this->repo->createPartida($idDocente, $codigo, $titulo, $duracion, $gradoObjetivo, $idCurso, $modoPreg, $modoAlt);
  }

  public function formPreguntas(int $idDocente, int $idPartida, string $grado = 'TODOS'): array {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $partida = $this->repo->findById($idPartida, $idDocente);
    if (!$partida) throw new Exception("Partida no encontrada.");

    $grado = strtoupper(trim($grado));
    if ($grado === '') $grado = 'TODOS';

    // ✅ si la partida tiene curso, filtramos por curso automáticamente
    $idCursoPartida = (int)($partida['id_curso'] ?? 0);

    if ($idCursoPartida > 0) {
      $banco = $this->pregRepo->getBancoByDocenteCursoGrado($idDocente, $idCursoPartida, $grado);
    } else {
      $banco = $this->pregRepo->getBancoByDocenteGrado($idDocente, $grado);
    }

    $seleccionadas = $this->repo->getPreguntasIds($idPartida);

    $mapOrden = [];
    $mapPuntos = [];
    foreach ($seleccionadas as $s) {
      $idp = (int)$s['id_pregunta'];
      $mapOrden[$idp]  = (int)$s['orden'];
      $mapPuntos[$idp] = (int)($s['puntos'] ?? 1);
    }

    return [
      "partida" => $partida,
      "banco" => $banco,
      "grado" => $grado,
      "mapOrden" => $mapOrden,
      "mapPuntos" => $mapPuntos,
    ];
  }

  public function guardarPreguntas(int $idDocente, int $idPartida, array $data): void {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $partida = $this->repo->findById($idPartida, $idDocente);
    if (!$partida) throw new Exception("Partida no encontrada.");

    $seleccion = $data['sel'] ?? [];
    $ordenes   = $data['orden'] ?? [];
    $puntosArr = $data['puntos'] ?? [];

    if (empty($seleccion)) throw new Exception("Selecciona al menos 1 pregunta.");

    $items = [];
    foreach ($seleccion as $idPreguntaStr => $_) {
      $idPregunta = (int)$idPreguntaStr;

      $orden = (int)($ordenes[$idPreguntaStr] ?? 0);
      if ($orden <= 0) throw new Exception("Cada pregunta seleccionada debe tener un orden > 0.");

      $puntos = (int)($puntosArr[$idPreguntaStr] ?? 1);
      if ($puntos < 1 || $puntos > 100) throw new Exception("Puntos inválidos (1..100).");

      $items[] = ["id"=>$idPregunta, "orden"=>$orden, "puntos"=>$puntos];
    }

    $ordList = array_map(fn($x)=>$x["orden"], $items);
    if (count($ordList) !== count(array_unique($ordList))) {
      throw new Exception("No repitas el orden. Debe ser 1,2,3...");
    }

    usort($items, fn($a,$b)=>$a["orden"] <=> $b["orden"]);

    $pdo = db();
    $pdo->beginTransaction();
    try {
      $this->repo->clearPreguntas($idPartida);
      foreach ($items as $it) {
        $this->repo->addPreguntaToPartida($idPartida, $it["id"], $it["orden"], $it["puntos"]);
      }
      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function iniciar(int $idDocente, int $idPartida): void {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");
    if (($p['estado'] ?? '') !== 'CREADA') throw new Exception("Solo se puede iniciar si está CREADA.");

    if ($this->repo->countPreguntas($idPartida) <= 0) {
      throw new Exception("Agrega preguntas antes de iniciar.");
    }

    $this->repo->setEstadoIniciar($idPartida, $idDocente);
  }

  public function finalizar(int $idDocente, int $idPartida): void {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");
    if (($p['estado'] ?? '') !== 'EN_CURSO') throw new Exception("Solo se puede finalizar si está EN_CURSO.");

    $this->repo->setEstadoFinalizar($idPartida, $idDocente);
  }

  public function ranking(int $idDocente, int $idPartida): array {
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");
    return $this->repo->getRanking($idPartida);
  }

  public function controlData(int $idDocente, int $idPartida): array {
    // ✅ si venció, la FINALIZA automáticamente
    $this->repo->autoFinalizarSiVencio($idPartida);

    $p = $this->repo->findById($idPartida, $idDocente);
    if (!$p) throw new Exception("Partida no encontrada.");

    $totalPreg = $this->repo->countTotalPreguntas($idPartida);
    $seg = $this->repo->getSegundosRestantes($idPartida);
    if ($seg < 0) $seg = 0;

    $resumen = $this->repo->getControlResumen($idPartida);
    $lista = $this->repo->getControlParticipantes($idPartida);

    foreach ($lista as &$x) {
      $resp = (int)($x["respuestas_registradas"] ?? 0);
      $pct = ($totalPreg > 0) ? (int)round(($resp / $totalPreg) * 100) : 0;
      $x["total_preguntas"] = $totalPreg;
      $x["progreso_pct"] = $pct;
    }
    unset($x);

    $top = $this->repo->getRanking($idPartida);
    $top = array_slice($top, 0, 10);

    return [
      "partida" => [
        "id_partida" => (int)$p["id_partida"],
        "codigo" => (string)$p["codigo"],
        "titulo" => (string)$p["titulo"],
        "estado" => (string)$p["estado"],
        "id_curso" => $p["id_curso"] ?? null,
      ],
      "tiempo" => [
        "segundos_restantes" => (int)$seg,
        "total_preguntas" => (int)$totalPreg,
      ],
      "resumen" => $resumen,
      "participantes" => $lista,
      "top" => $top,
    ];
  }
}
