<?php
declare(strict_types=1);

require_once APP_PATH . "/repositories/PartidaRepository.php";
require_once APP_PATH . "/repositories/ParticipanteRepository.php";
require_once APP_PATH . "/repositories/ComputadoraRepository.php";

class ParticipacionService {
  private PartidaRepository $partidaRepo;
  private ParticipanteRepository $partRepo;

  /** Convierte el grado del estudiante (string) a entero 0..5. 0 = no definido/TODOS */
  private function gradoToInt(string $grado): int {
    $g = strtoupper(trim($grado));
    if ($g === '' || $g === 'TODOS') return 0;
    if (preg_match('/(\d+)/', $g, $m)) {
      $n = (int)$m[1];
      return ($n >= 1 && $n <= 5) ? $n : 0;
    }
    return 0;
  }

  public function __construct() {
    $this->partidaRepo = new PartidaRepository();
    $this->partRepo = new ParticipanteRepository();
  }

  public function unirse(int $idEstudiante, string $codigo, string $aliasBase, string $ip = ''): array {
    $codigo = strtoupper(trim($codigo));
    $aliasBase = trim($aliasBase);

    if ($codigo === '') {
      throw new Exception("El código es obligatorio.");
    }
    if ($aliasBase !== '' && mb_strlen($aliasBase) > 60) {
      throw new Exception("Nombre demasiado largo (máx 60).");
    }

    // Safety: en algunos entornos se han reportado instancias null por autoload/estado.
    // Re-inicializamos repositorios para evitar 'Call to a member function ... on null'.
    $this->partidaRepo = new PartidaRepository();
    $this->partRepo = new ParticipanteRepository();

    $partida = $this->partidaRepo->findByCodigo($codigo);
    if (!$partida) throw new Exception("Código de partida no existe.");

    // Validación por grado objetivo (clave del sistema)
    // 0 = TODOS (cualquiera entra). 1..5 = solo ese grado.
    $gradoObjetivo = (int)($partida['grado_objetivo'] ?? 0);
    if ($gradoObjetivo < 0 || $gradoObjetivo > 5) $gradoObjetivo = 0;
    if ($gradoObjetivo > 0) {
      $pdoG = db();
      $stmtG = $pdoG->prepare("SELECT grado FROM estudiante WHERE id_usuario = ? LIMIT 1");
      $stmtG->execute([$idEstudiante]);
      $rowG = $stmtG->fetch(PDO::FETCH_ASSOC);
      $gradoEstInt = $this->gradoToInt($rowG ? (string)($rowG['grado'] ?? '') : '');
      if ($gradoEstInt !== $gradoObjetivo) {
        throw new Exception("No puedes unirte: esta partida es solo para el grado {$gradoObjetivo}.");
      }
    }

    if ($partida['estado'] === 'FINALIZADA') {
      throw new Exception("La partida ya finalizó.");
    }

    $idPartida = (int)$partida['id_partida'];

    // Alias automático: usar nombre real del estudiante (y asegurar unicidad).
    $alias = $aliasBase !== '' ? $aliasBase : ('EST-' . $idEstudiante);
    $alias = trim($alias);
    if ($alias === '') { $alias = 'EST-' . $idEstudiante; }
    if (mb_strlen($alias) > 60) { $alias = mb_substr($alias, 0, 60); }

    // Si ya se unió antes y NO finalizó, reusa ese intento.
    // Si ya finalizó (o quedó grabado por pruebas anteriores), se crea un nuevo intento,
    // para evitar que el estudiante entre y le salga "ya diste el examen" sin responder.
    $ya = $this->partRepo->findByPartidaAndEstudiante($idPartida, $idEstudiante);
    if ($ya && (int)($ya['finalizado'] ?? 0) === 0) {
      if ($ip !== '') {
        try { (new ComputadoraRepository())->touchOcupadaByIp($ip); } catch (Throwable $e2) {}
      }
      return [
        "partida" => $partida,
        "participante" => [
          "id_participante" => (int)$ya["id_participante"],
          "alias" => $ya["alias"]
        ],
        "ip" => $ip
      ];
    }

    // Si la IP está registrada, marcamos la PC como ocupada al crear el nuevo intento.
    if ($ip !== '') {
      try { (new ComputadoraRepository())->touchOcupadaByIp($ip); } catch (Throwable $e3) {}
    }

    // Alias único por partida (si existe, se agrega sufijo automático)
    $aliasBase2 = $alias;
    $i = 2;
    while ($this->partRepo->aliasExisteEnPartida($idPartida, $alias)) {
      $alias = $aliasBase2 . '-' . $i;
      $i++;
      if ($i > 99) {
        throw new Exception("No se pudo generar un alias disponible.");
      }
    }

    $pdo = db();

    $pdo->beginTransaction();
    try {
      $idParticipante = $this->partRepo->create($idPartida, $idEstudiante, $alias);
      $pdo->commit();

      // Marcar computadora OCUPADA por IP (si está registrada)
      if ($ip !== '') {
        try {
          (new ComputadoraRepository())->touchOcupadaByIp($ip);
        } catch (Throwable $e2) {
          // si no existe la PC, no bloqueamos el flujo
        }
      }

      return [
        "partida" => $partida,
        "participante" => [
          "id_participante" => $idParticipante,
          "alias" => $alias
        ],
        "ip" => $ip
      ];
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }
}