<?php
declare(strict_types=1);

require_once APP_PATH . "/repositories/EstudianteRepository.php";

class EstudianteService {
  private EstudianteRepository $repo;

  public function __construct() {
    $this->repo = new EstudianteRepository();
  }

  public function listar(): array {
    return $this->repo->getAll();
  }

  public function registrar(array $data): void {
    $dni = trim($data['dni'] ?? '');
    $nombres = trim($data['nombres'] ?? '');
    $apellidos = trim($data['apellidos'] ?? '');
    $grado = trim($data['grado'] ?? '');
    $seccion = trim($data['seccion'] ?? '');

    if ($dni === '' || $nombres === '' || $apellidos === '' || $grado === '' || $seccion === '') {
      throw new Exception("Completa todos los campos.");
    }

    if (!preg_match('/^[0-9]{8,12}$/', $dni)) {
      throw new Exception("DNI inválido (solo números, 8 a 12 dígitos).");
    }

    // username = DNI
    $username = $dni;

    if ($this->repo->existsDni($dni)) {
      throw new Exception("Ya existe un estudiante con ese DNI.");
    }

    if ($this->repo->existsUsername($username)) {
      throw new Exception("Ya existe un usuario con ese username (DNI).");
    }

    // contraseña inicial = DNI
    $passwordHash = password_hash($dni, PASSWORD_BCRYPT);

    $pdo = db();
    $pdo->beginTransaction();
    try {
      $idUsuario = $this->repo->createUser($nombres, $apellidos, $username, $passwordHash);
      $this->repo->createEstudiante($idUsuario, $dni, $grado, $seccion);
      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function obtener(int $idUsuario): array {
    $e = $this->repo->getByUsuarioId($idUsuario);
    if (!$e) {
      throw new Exception('Estudiante no encontrado.');
    }
    return $e;
  }

  public function actualizar(int $idUsuario, array $data): void {
    $dni = trim($data['dni'] ?? '');
    $nombres = trim($data['nombres'] ?? '');
    $apellidos = trim($data['apellidos'] ?? '');
    $grado = trim($data['grado'] ?? '');
    $seccion = trim($data['seccion'] ?? '');
    $estado = strtoupper(trim($data['estado'] ?? 'ACTIVO'));

    if ($dni === '' || $nombres === '' || $apellidos === '' || $grado === '' || $seccion === '') {
      throw new Exception('Completa todos los campos.');
    }
    if (!preg_match('/^[0-9]{8,12}$/', $dni)) {
      throw new Exception('DNI inválido (solo números, 8 a 12 dígitos).');
    }
    if (!in_array($estado, ['ACTIVO','INACTIVO'], true)) {
      $estado = 'ACTIVO';
    }

    $actual = $this->repo->getByUsuarioId($idUsuario);
    if (!$actual) throw new Exception('Estudiante no encontrado.');

    // Si cambia DNI, también cambia username (username = DNI)
    $username = $dni;

    if ($dni !== $actual['dni'] && $this->repo->existsDni($dni)) {
      throw new Exception('Ya existe un estudiante con ese DNI.');
    }
    if ($username !== $actual['username'] && $this->repo->existsUsername($username)) {
      throw new Exception('Ya existe un usuario con ese username (DNI).');
    }

    $pdo = db();
    $pdo->beginTransaction();
    try {
      $this->repo->updateUser($idUsuario, $nombres, $apellidos, $username, $estado);
      $this->repo->updateEstudiante($idUsuario, $dni, $grado, $seccion);
      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function eliminar(int $idUsuario): void {
    $actual = $this->repo->getByUsuarioId($idUsuario);
    if (!$actual) throw new Exception('Estudiante no encontrado.');

    $pdo = db();
    $pdo->beginTransaction();
    try {
      // Primero borrar de estudiante, luego usuario
      $this->repo->deleteEstudiante($idUsuario);
      $this->repo->deleteUsuario($idUsuario);
      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }
}
