<?php
declare(strict_types=1);

class AdminDocenteService {
  private DocenteRepository $repo;

  public function __construct() {
    require_once APP_PATH . "/repositories/DocenteRepository.php";
    $this->repo = new DocenteRepository();
  }

  public function listar(): array {
    return $this->repo->listAll();
  }

  public function eliminar(int $idUsuario): void {
    if ($idUsuario <= 0) throw new Exception('Docente inválido.');
    $this->repo->deleteById($idUsuario);
  }

  public function buscar(int $idUsuario): ?array {
    return $this->repo->findById($idUsuario);
  }

  public function registrar(array $data): void {
    $nombres = trim((string)($data['nombres'] ?? ''));
    $apellidos = trim((string)($data['apellidos'] ?? ''));
    $username = trim((string)($data['username'] ?? ''));
    $especialidad = trim((string)($data['especialidad'] ?? ''));
    $claveInicial = trim((string)($data['clave_inicial'] ?? ''));

    if ($nombres === '' || $apellidos === '' || $username === '') {
      throw new Exception('Completa nombres, apellidos y username.');
    }

    if (strlen($username) < 3) {
      throw new Exception('El username debe tener al menos 3 caracteres.');
    }

    // clave por defecto: username
    if ($claveInicial === '') {
      $claveInicial = $username;
    }

    $this->repo->create([
      'nombres' => $nombres,
      'apellidos' => $apellidos,
      'username' => $username,
      'especialidad' => $especialidad,
      'clave_inicial' => $claveInicial,
    ]);
  }

  public function actualizar(array $data): void {
    $id = (int)($data['id_usuario'] ?? 0);
    $nombres = trim((string)($data['nombres'] ?? ''));
    $apellidos = trim((string)($data['apellidos'] ?? ''));
    $username = trim((string)($data['username'] ?? ''));
    $especialidad = trim((string)($data['especialidad'] ?? ''));

    if ($id <= 0) {
      throw new Exception('Docente inválido.');
    }
    if ($nombres === '' || $apellidos === '' || $username === '') {
      throw new Exception('Completa nombres, apellidos y username.');
    }

    $this->repo->update([
      'id_usuario' => $id,
      'nombres' => $nombres,
      'apellidos' => $apellidos,
      'username' => $username,
      'especialidad' => $especialidad,
    ]);
  }

  public function cambiarEstado(int $idUsuario, string $estadoActual): void {
    if ($idUsuario <= 0) throw new Exception('Docente inválido.');

    $estadoActual = strtoupper(trim($estadoActual));
    $nuevo = ($estadoActual === 'ACTIVO') ? 'INACTIVO' : 'ACTIVO';

    $this->repo->setEstado($idUsuario, $nuevo);
  }

  /**
   * Resetea contraseña.
   * - Si el admin no manda nueva_clave, se usa username como clave inicial.
   * - Devuelve la clave final para mostrarla en alerta.
   */
  public function resetearClave(int $idUsuario, string $nuevaClave): string {
    if ($idUsuario <= 0) throw new Exception('Docente inválido.');

    $doc = $this->repo->findById($idUsuario);
    if (!$doc) throw new Exception('Docente no encontrado.');

    $pass = trim($nuevaClave);
    if ($pass === '') {
      $pass = (string)$doc['username'];
    }

    if (strlen($pass) < 4) {
      throw new Exception('La clave debe tener al menos 4 caracteres.');
    }

    $this->repo->resetPassword($idUsuario, $pass);
    return $pass;
  }
}
