<?php
declare(strict_types=1);

class QuizRepository {

  /** Lista todas las preguntas de una partida (ordenadas) */
  public function listPreguntas(int $idPartida): array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT pp.id AS id_partida_pregunta, pp.orden,
              q.id_pregunta, q.enunciado, q.dificultad
       FROM partida_pregunta pp
       JOIN pregunta q ON q.id_pregunta = pp.id_pregunta
       WHERE pp.id_partida = ?
       ORDER BY pp.orden ASC"
    );
    $st->execute([$idPartida]);
    return $st->fetchAll();
  }

  /** Trae una pregunta por orden dentro de la partida */
  public function getPreguntaByOrden(int $idPartida, int $orden): ?array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT pp.id AS id_partida_pregunta, pp.orden,
              q.id_pregunta, q.enunciado, q.dificultad
       FROM partida_pregunta pp
       JOIN pregunta q ON q.id_pregunta = pp.id_pregunta
       WHERE pp.id_partida = ? AND pp.orden = ?
       LIMIT 1"
    );
    $st->execute([$idPartida, $orden]);
    $row = $st->fetch();
    return $row ?: null;
  }

  public function getAlternativas(int $idPregunta): array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT id_alternativa, texto
       FROM alternativa
       WHERE id_pregunta = ?
       ORDER BY id_alternativa ASC"
    );
    $st->execute([$idPregunta]);
    return $st->fetchAll();
  }

  /**
   * Upsert de respuesta (permite cambiar). Recalcula puntaje_obtenido según alternativa.
   * Requiere UNIQUE(id_participante, id_partida_pregunta) en respuesta.
   */
  public function upsertRespuesta(int $idParticipante, int $idPartidaPregunta, int $idAlternativa, float $tiempo): void {
    $pdo = db();

    // ✅ FASE 4: puntaje por pregunta (puntos definidos en partida_pregunta.puntos)
    // - Si la alternativa es correcta: suma puntos de esa pregunta.
    // - Si no es correcta: 0.
    // Repetimos parámetros para soportar MariaDB/MySQL (sin VALUES() en algunos entornos).
    $sql = "INSERT INTO respuesta (
              id_participante, id_partida_pregunta, id_alternativa, tiempo_respuesta, puntaje_obtenido, fecha_respuesta
            )
            VALUES (
              ?, ?, ?, ?,
              IF(
                (SELECT es_correcta FROM alternativa WHERE id_alternativa = ? LIMIT 1) = 1,
                (SELECT puntos FROM partida_pregunta WHERE id = ? LIMIT 1),
                0
              ),
              NOW()
            )
            ON DUPLICATE KEY UPDATE
              id_alternativa = ?,
              tiempo_respuesta = ?,
              puntaje_obtenido = IF(
                (SELECT es_correcta FROM alternativa WHERE id_alternativa = ? LIMIT 1) = 1,
                (SELECT puntos FROM partida_pregunta WHERE id = ? LIMIT 1),
                0
              ),
              fecha_respuesta = NOW()";

    $st = $pdo->prepare($sql);
    $st->execute([
      $idParticipante,
      $idPartidaPregunta,
      $idAlternativa,
      $tiempo,
      $idAlternativa,     // IF INSERT: alternativa
      $idPartidaPregunta, // IF INSERT: puntos
      $idAlternativa,     // UPDATE id_alternativa
      $tiempo,            // UPDATE tiempo_respuesta
      $idAlternativa,     // IF UPDATE: alternativa
      $idPartidaPregunta  // IF UPDATE: puntos
    ]);
  }



  /** Mapa de respuestas del participante en una partida: [id_partida_pregunta => id_alternativa] */
  public function getRespuestasMap(int $idPartida, int $idParticipante): array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT r.id_partida_pregunta, r.id_alternativa
       FROM respuesta r
       JOIN partida_pregunta pp ON pp.id = r.id_partida_pregunta
       WHERE pp.id_partida = ? AND r.id_participante = ?"
    );
    $st->execute([$idPartida, $idParticipante]);
    $rows = $st->fetchAll();
    $map = [];
    foreach ($rows as $r) {
      $map[(int)$r['id_partida_pregunta']] = (int)$r['id_alternativa'];
    }
    return $map;
  }

  public function countRespondidas(int $idPartida, int $idParticipante): int {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT COUNT(*) c
       FROM respuesta r
       JOIN partida_pregunta pp ON pp.id = r.id_partida_pregunta
       WHERE pp.id_partida = ? AND r.id_participante = ?"
    );
    $st->execute([$idPartida, $idParticipante]);
    return (int)$st->fetch()['c'];
  }

  public function countTotalPreguntas(int $idPartida): int {
    $pdo = db();
    $st = $pdo->prepare("SELECT COUNT(*) c FROM partida_pregunta WHERE id_partida=?");
    $st->execute([$idPartida]);
    return (int)$st->fetch()['c'];
  }
}
