<?php
declare(strict_types=1);

class PreguntaRepository {

  public function getAllByDocente(int $idDocente): array {
    $pdo = db();
    $st = $pdo->prepare("
      SELECT id_pregunta, enunciado, dificultad, activo, created_at, grado, imagen, id_curso
      FROM pregunta
      WHERE id_docente = ?
      ORDER BY created_at DESC
    ");
    $st->execute([$idDocente]);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  /**
   * Banco filtrado por grado.
   * - Si $grado = 'TODOS' o vacío: devuelve todo.
   * - Si $grado es 1..5: devuelve (grado = $grado) OR (grado = 'TODOS')
   */
  public function getBancoByDocenteGrado(int $idDocente, string $grado = 'TODOS'): array {
    $pdo = db();
    $grado = strtoupper(trim($grado));

    if ($grado === '' || $grado === 'TODOS') {
      return $this->getAllByDocente($idDocente);
    }

    $st = $pdo->prepare("
      SELECT id_pregunta, enunciado, dificultad, activo, created_at, grado, imagen, id_curso
      FROM pregunta
      WHERE id_docente = ?
        AND (grado = ? OR grado = 'TODOS')
      ORDER BY created_at DESC
    ");
    $st->execute([$idDocente, $grado]);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  public function createPregunta(
    string $enunciado,
    string $dificultad,
    int $idDocente,
    string $grado,
    ?int $idCurso,
    ?string $imagen
  ): int {
    $pdo = db();
    $st = $pdo->prepare("
      INSERT INTO pregunta (enunciado, imagen, dificultad, activo, id_docente, grado, id_curso)
      VALUES (?, ?, ?, 1, ?, ?, ?)
    ");
    $st->execute([$enunciado, $imagen, $dificultad, $idDocente, $grado, $idCurso]);
    return (int)$pdo->lastInsertId();
  }

  public function addAlternativa(int $idPregunta, string $texto, bool $esCorrecta): void {
    $pdo = db();
    $st = $pdo->prepare("
      INSERT INTO alternativa (id_pregunta, texto, es_correcta)
      VALUES (?, ?, ?)
    ");
    $st->execute([$idPregunta, $texto, $esCorrecta ? 1 : 0]);
  }

  public function getAlternativas(int $idPregunta): array {
    $pdo = db();
    $st = $pdo->prepare("
      SELECT id_alternativa, texto, es_correcta
      FROM alternativa
      WHERE id_pregunta = ?
      ORDER BY id_alternativa ASC
    ");
    $st->execute([$idPregunta]);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  public function findById(int $idPregunta, int $idDocente): ?array {
    $pdo = db();
    $st = $pdo->prepare("
      SELECT id_pregunta, enunciado, dificultad, activo, grado, imagen, id_curso
      FROM pregunta
      WHERE id_pregunta = ? AND id_docente = ?
      LIMIT 1
    ");
    $st->execute([$idPregunta, $idDocente]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
  }

  public function updatePregunta(
    int $idPregunta,
    int $idDocente,
    string $enunciado,
    string $dificultad,
    string $grado,
    int $activo,
    ?int $idCurso,
    ?string $imagen = null
  ): void {
    $pdo = db();

    if ($imagen !== null) {
      $st = $pdo->prepare("
        UPDATE pregunta
        SET enunciado=?, dificultad=?, grado=?, activo=?, id_curso=?, imagen=?
        WHERE id_pregunta=? AND id_docente=?
      ");
      $st->execute([$enunciado, $dificultad, $grado, $activo, $idCurso, $imagen, $idPregunta, $idDocente]);
      return;
    }

    $st = $pdo->prepare("
      UPDATE pregunta
      SET enunciado=?, dificultad=?, grado=?, activo=?, id_curso=?
      WHERE id_pregunta=? AND id_docente=?
    ");
    $st->execute([$enunciado, $dificultad, $grado, $activo, $idCurso, $idPregunta, $idDocente]);
  }

  public function deleteAlternativas(int $idPregunta): void {
    $pdo = db();
    $st = $pdo->prepare("DELETE FROM alternativa WHERE id_pregunta = ?");
    $st->execute([$idPregunta]);
  }

  public function deletePregunta(int $idPregunta, int $idDocente): void {
    $pdo = db();
    $st = $pdo->prepare("DELETE FROM pregunta WHERE id_pregunta = ? AND id_docente = ?");
    $st->execute([$idPregunta, $idDocente]);
  }

  public function getAllByDocenteCurso(int $idDocente, int $idCurso): array {
    $pdo = db();
    $st = $pdo->prepare("
      SELECT id_pregunta, enunciado, dificultad, activo, created_at, grado, imagen, id_curso
      FROM pregunta
      WHERE id_docente = ? AND id_curso = ?
      ORDER BY created_at DESC
    ");
    $st->execute([$idDocente, $idCurso]);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

    public function getBancoByDocenteCursoGrado(int $idDocente, int $idCurso, string $grado = 'TODOS'): array {
  $pdo = db();
  $grado = strtoupper(trim($grado));

  if ($grado === '' || $grado === 'TODOS') {
    $st = $pdo->prepare("
      SELECT id_pregunta, enunciado, dificultad, activo, created_at, grado, imagen
      FROM pregunta
      WHERE id_docente = ? AND id_curso = ?
      ORDER BY created_at DESC
    ");
    $st->execute([$idDocente, $idCurso]);
    return $st->fetchAll() ?: [];
  }

  $st = $pdo->prepare("
    SELECT id_pregunta, enunciado, dificultad, activo, created_at, grado, imagen
    FROM pregunta
    WHERE id_docente = ? AND id_curso = ?
      AND (grado = ? OR grado = 'TODOS')
    ORDER BY created_at DESC
  ");
  $st->execute([$idDocente, $idCurso, $grado]);
  return $st->fetchAll() ?: [];
}


}
