<?php
declare(strict_types=1);

class PartidaRepository {

  // ✅ AUTO FINALIZA TODAS LAS VENCIDAS DEL DOCENTE (para listar)
  public function autoFinalizarVencidasPorDocente(int $idDocente): void {
    $pdo = db();
    $st = $pdo->prepare(
      "UPDATE partida
       SET estado='FINALIZADA'
       WHERE id_docente = ?
         AND estado='EN_CURSO'
         AND fecha_fin IS NOT NULL
         AND fecha_fin <= NOW()"
    );
    $st->execute([$idDocente]);
  }

  // ✅ AUTO FINALIZA UNA PARTIDA SI YA VENCIO (para control/ranking/etc.)
  public function autoFinalizarSiVencio(int $idPartida): void {
    $pdo = db();
    $st = $pdo->prepare(
      "UPDATE partida
       SET estado='FINALIZADA'
       WHERE id_partida = ?
         AND estado='EN_CURSO'
         AND fecha_fin IS NOT NULL
         AND fecha_fin <= NOW()"
    );
    $st->execute([$idPartida]);
  }

  public function getAllByDocenteFiltrado(
    int $idDocente,
    ?int $idCurso,
    ?string $estado,
    ?int $grado,
    ?string $desde,
    ?string $hasta
  ): array {
    $pdo = db();

    $sql = "SELECT id_partida, codigo, titulo, estado, duracion_total_min, grado_objetivo, id_curso, created_at
            FROM partida
            WHERE id_docente = :doc";
    $params = [":doc" => $idDocente];

    if ($idCurso && $idCurso > 0) { $sql .= " AND id_curso = :curso"; $params[":curso"] = $idCurso; }
    if ($estado) { $sql .= " AND estado = :estado"; $params[":estado"] = $estado; }
    if ($grado !== null && $grado >= 0 && $grado <= 5) { $sql .= " AND grado_objetivo = :grado"; $params[":grado"] = $grado; }

    if ($desde) { $sql .= " AND DATE(created_at) >= :desde"; $params[":desde"] = $desde; }
    if ($hasta) { $sql .= " AND DATE(created_at) <= :hasta"; $params[":hasta"] = $hasta; }

    $sql .= " ORDER BY created_at DESC";

    $st = $pdo->prepare($sql);
    $st->execute($params);
    return $st->fetchAll() ?: [];
  }

  public function createPartida(
    int $idDocente,
    string $codigo,
    string $titulo,
    int $duracionMin,
    int $gradoObjetivo,
    ?int $idCurso,
    string $modoPreguntas,
    string $modoAlternativas
  ): int {
    $pdo = db();
    $st = $pdo->prepare(
      "INSERT INTO partida (codigo, titulo, estado, duracion_total_min, id_docente, grado_objetivo, grado, id_curso, modo_preguntas, modo_alternativas)
       VALUES (?, ?, 'CREADA', ?, ?, ?, ?, ?, ?, ?)"
    );
    $st->execute([
      $codigo,
      $titulo,
      $duracionMin,
      $idDocente,
      $gradoObjetivo,
      $gradoObjetivo,
      $idCurso,
      $modoPreguntas,
      $modoAlternativas
    ]);
    return (int)$pdo->lastInsertId();
  }

  public function updatePartida(
    int $idPartida,
    int $idDocente,
    string $titulo,
    int $duracionMin,
    int $gradoObjetivo,
    ?int $idCurso,
    string $modoPreguntas,
    string $modoAlternativas
  ): void {
    $pdo = db();
    $st = $pdo->prepare(
      "UPDATE partida
       SET titulo=?, duracion_total_min=?, grado_objetivo=?, grado=?, id_curso=?, modo_preguntas=?, modo_alternativas=?
       WHERE id_partida=? AND id_docente=? AND estado='CREADA'"
    );
    $st->execute([
      $titulo, $duracionMin, $gradoObjetivo, $gradoObjetivo,
      $idCurso, $modoPreguntas, $modoAlternativas,
      $idPartida, $idDocente
    ]);
  }

  public function deletePartida(int $idPartida, int $idDocente): void {
    $pdo = db();
    $pdo->beginTransaction();
    try {
      $st = $pdo->prepare("DELETE FROM partida_pregunta WHERE id_partida=?");
      $st->execute([$idPartida]);

      $st = $pdo->prepare("DELETE FROM partida WHERE id_partida=? AND id_docente=? AND estado='CREADA'");
      $st->execute([$idPartida, $idDocente]);

      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function codigoExiste(string $codigo): bool {
    $pdo = db();
    $st = $pdo->prepare("SELECT 1 FROM partida WHERE codigo = ? LIMIT 1");
    $st->execute([$codigo]);
    return (bool)$st->fetch();
  }

  // ✅ ESTO YA EXISTE
  public function findById(int $idPartida, int $idDocente): ?array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT id_partida, codigo, titulo, estado, duracion_total_min, grado_objetivo,
              id_curso, modo_preguntas, modo_alternativas, created_at, fecha_inicio, fecha_fin
       FROM partida
       WHERE id_partida = ? AND id_docente = ?
       LIMIT 1"
    );
    $st->execute([$idPartida, $idDocente]);
    $row = $st->fetch();
    return $row ?: null;
  }

  // ✅ FIX: Alias para que NO reviente si tu controller llama obtener()
  // Si tu controller estaba llamando obtener($idPartida, $idDocente) => funciona
  // Si estaba llamando obtener($idPartida) => también funciona
  public function obtener(int $idPartida, ?int $idDocente = null): ?array {
    $pdo = db();

    if ($idDocente !== null) {
      return $this->findById($idPartida, $idDocente);
    }

    $st = $pdo->prepare(
      "SELECT id_partida, codigo, titulo, estado, duracion_total_min, grado_objetivo,
              id_curso, modo_preguntas, modo_alternativas, created_at, fecha_inicio, fecha_fin, id_docente
       FROM partida
       WHERE id_partida = ?
       LIMIT 1"
    );
    $st->execute([$idPartida]);
    $row = $st->fetch();
    return $row ?: null;
  }

  public function clearPreguntas(int $idPartida): void {
    $pdo = db();
    $st = $pdo->prepare("DELETE FROM partida_pregunta WHERE id_partida = ?");
    $st->execute([$idPartida]);
  }

  public function addPreguntaToPartida(int $idPartida, int $idPregunta, int $orden, int $puntos): void {
    $pdo = db();
    $st = $pdo->prepare(
      "INSERT INTO partida_pregunta (id_partida, id_pregunta, orden, puntos)
       VALUES (?, ?, ?, ?)"
    );
    $st->execute([$idPartida, $idPregunta, $orden, $puntos]);
  }

  public function getPreguntasIds(int $idPartida): array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT id_pregunta, orden, puntos
       FROM partida_pregunta
       WHERE id_partida = ?
       ORDER BY orden ASC"
    );
    $st->execute([$idPartida]);
    return $st->fetchAll() ?: [];
  }

  public function findByCodigo(string $codigo): ?array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT id_partida, codigo, titulo, estado, duracion_total_min, id_docente, fecha_inicio, fecha_fin, grado_objetivo, grado, id_curso
       FROM partida
       WHERE codigo = ?
       LIMIT 1"
    );
    $st->execute([$codigo]);
    $row = $st->fetch();
    return $row ?: null;
  }

  public function countPreguntas(int $idPartida): int {
    $pdo = db();
    $st = $pdo->prepare("SELECT COUNT(*) c FROM partida_pregunta WHERE id_partida=?");
    $st->execute([$idPartida]);
    return (int)$st->fetch()['c'];
  }

  public function setEstadoIniciar(int $idPartida, int $idDocente): void {
    $pdo = db();

    $pdo->beginTransaction();
    try {
      $st = $pdo->prepare(
        "DELETE r FROM respuesta r
         JOIN participante p ON p.id_participante = r.id_participante
         WHERE p.id_partida = ?"
      );
      $st->execute([$idPartida]);

      $st = $pdo->prepare("DELETE FROM participante WHERE id_partida = ?");
      $st->execute([$idPartida]);

      $st = $pdo->prepare(
        "UPDATE partida
         SET estado='EN_CURSO',
             fecha_inicio = NOW(),
             fecha_fin = DATE_ADD(NOW(), INTERVAL duracion_total_min MINUTE)
         WHERE id_partida=? AND id_docente=? AND estado='CREADA'"
      );
      $st->execute([$idPartida, $idDocente]);

      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function setEstadoFinalizar(int $idPartida, int $idDocente): void {
    $pdo = db();
    $st = $pdo->prepare(
      "UPDATE partida
       SET estado='FINALIZADA', fecha_fin = NOW()
       WHERE id_partida=? AND id_docente=? AND estado='EN_CURSO'"
    );
    $st->execute([$idPartida, $idDocente]);
  }

  public function getRanking(int $idPartida): array {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT alias, nombres, apellidos, puntaje_total, respuestas_registradas
       FROM vw_ranking_partida
       WHERE id_partida = ?
       ORDER BY puntaje_total DESC, respuestas_registradas DESC, alias ASC"
    );
    $st->execute([$idPartida]);
    return $st->fetchAll() ?: [];
  }

  public function getEstadoById(int $idPartida): ?string {
    $pdo = db();
    $st = $pdo->prepare("SELECT estado FROM partida WHERE id_partida=? LIMIT 1");
    $st->execute([$idPartida]);
    $row = $st->fetch();
    return $row ? (string)$row['estado'] : null;
  }

  public function getTiempoData(int $idPartida): ?array {
    $pdo = db();
    $st = $pdo->prepare("SELECT duracion_total_min, fecha_inicio, fecha_fin, estado FROM partida WHERE id_partida=? LIMIT 1");
    $st->execute([$idPartida]);
    $row = $st->fetch();
    return $row ?: null;
  }

  public function getSegundosRestantes(int $idPartida): int {
    $pdo = db();
    $st = $pdo->prepare(
      "SELECT 
          CASE 
            WHEN fecha_fin IS NULL THEN 0
            ELSE TIMESTAMPDIFF(SECOND, NOW(), fecha_fin)
          END AS restante
        FROM partida
        WHERE id_partida = ?
        LIMIT 1"
    );
    $st->execute([$idPartida]);
    $row = $st->fetch();
    if (!$row || !isset($row['restante'])) return 0;
    return (int)$row['restante'];
  }

  public function countTotalPreguntas(int $idPartida): int {
    $pdo = db();
    $st = $pdo->prepare("SELECT COUNT(*) c FROM partida_pregunta WHERE id_partida=?");
    $st->execute([$idPartida]);
    $row = $st->fetch();
    return (int)($row['c'] ?? 0);
  }

  public function getControlParticipantes(int $idPartida): array {
    $pdo = db();

    $st = $pdo->prepare("
      SELECT
        p.id_participante,
        p.alias,
        u.nombres,
        u.apellidos,
        p.finalizado,
        p.fecha_union,
        p.finalizado_at,

        COALESCE(SUM(r.puntaje_obtenido),0) AS puntaje_total,
        COUNT(r.id_respuesta) AS respuestas_registradas,
        MAX(r.fecha_respuesta) AS ultima_respuesta

      FROM participante p
      JOIN usuario u ON u.id_usuario = p.id_estudiante
      LEFT JOIN respuesta r ON r.id_participante = p.id_participante
      WHERE p.id_partida = ?
      GROUP BY p.id_participante, p.alias, u.nombres, u.apellidos, p.finalizado, p.fecha_union, p.finalizado_at
      ORDER BY puntaje_total DESC, respuestas_registradas DESC, p.alias ASC
    ");
    $st->execute([$idPartida]);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  public function getControlResumen(int $idPartida): array {
    $pdo = db();
    $st = $pdo->prepare("
      SELECT
        COUNT(*) AS conectados,
        SUM(CASE WHEN finalizado = 1 THEN 1 ELSE 0 END) AS finalizados
      FROM participante
      WHERE id_partida = ?
    ");
    $st->execute([$idPartida]);
    $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    return [
      "conectados" => (int)($row["conectados"] ?? 0),
      "finalizados" => (int)($row["finalizados"] ?? 0),
    ];
  }
}
