<?php
declare(strict_types=1);

class DocenteRepository {

  public function listAll(): array {
    $pdo = db();
    $st = $pdo->query("
      SELECT u.id_usuario, u.nombres, u.apellidos, u.username, u.estado, u.created_at,
             d.especialidad
      FROM docente d
      JOIN usuario u ON u.id_usuario = d.id_usuario
      ORDER BY u.created_at DESC
    ");
    return $st->fetchAll();
  }

  public function findById(int $idUsuario): ?array {
    $pdo = db();
    $st = $pdo->prepare("
      SELECT u.id_usuario, u.nombres, u.apellidos, u.username, u.estado, u.created_at,
             d.especialidad
      FROM docente d
      JOIN usuario u ON u.id_usuario = d.id_usuario
      WHERE u.id_usuario = ?
      LIMIT 1
    ");
    $st->execute([$idUsuario]);
    $row = $st->fetch();
    return $row ?: null;
  }

  private function usernameExists(string $username, ?int $excludeId = null): bool {
    $pdo = db();
    if ($excludeId) {
      $st = $pdo->prepare("SELECT 1 FROM usuario WHERE username=? AND id_usuario<>? LIMIT 1");
      $st->execute([$username, $excludeId]);
    } else {
      $st = $pdo->prepare("SELECT 1 FROM usuario WHERE username=? LIMIT 1");
      $st->execute([$username]);
    }
    return (bool)$st->fetch();
  }

  public function create(array $data): void {
    $pdo = db();
    $pdo->beginTransaction();
    try {
      $username = (string)$data['username'];
      if ($this->usernameExists($username, null)) {
        throw new Exception('El username ya existe. Usa otro.');
      }

      $hash = password_hash((string)$data['clave_inicial'], PASSWORD_DEFAULT);

      $st = $pdo->prepare("
        INSERT INTO usuario (nombres, apellidos, username, password_hash, estado)
        VALUES (?, ?, ?, ?, 'ACTIVO')
      ");
      $st->execute([
        $data['nombres'],
        $data['apellidos'],
        $username,
        $hash
      ]);

      $idUsuario = (int)$pdo->lastInsertId();

      $st = $pdo->prepare("INSERT INTO docente (id_usuario, especialidad) VALUES (?, ?)");
      $st->execute([$idUsuario, $data['especialidad'] ?: null]);

      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function update(array $data): void {
    $pdo = db();
    $pdo->beginTransaction();
    try {
      $id = (int)$data['id_usuario'];
      $current = $this->findById($id);
      if (!$current) {
        throw new Exception('Docente no encontrado.');
      }

      $username = (string)$data['username'];
      if ($this->usernameExists($username, $id)) {
        throw new Exception('El username ya existe. Usa otro.');
      }

      $st = $pdo->prepare("
        UPDATE usuario
        SET nombres=?, apellidos=?, username=?
        WHERE id_usuario=?
      ");
      $st->execute([
        $data['nombres'],
        $data['apellidos'],
        $username,
        $id
      ]);

      $st = $pdo->prepare("UPDATE docente SET especialidad=? WHERE id_usuario=?");
      $st->execute([$data['especialidad'] ?: null, $id]);

      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }

  public function setEstado(int $idUsuario, string $estado): void {
    // aseguramos que sea docente
    if (!$this->findById($idUsuario)) {
      throw new Exception('Docente no encontrado.');
    }
    $pdo = db();
    $st = $pdo->prepare("UPDATE usuario SET estado=? WHERE id_usuario=?");
    $st->execute([strtoupper($estado), $idUsuario]);
  }

  public function resetPassword(int $idUsuario, string $plainPassword): void {
    if (!$this->findById($idUsuario)) {
      throw new Exception('Docente no encontrado.');
    }
    $hash = password_hash($plainPassword, PASSWORD_DEFAULT);
    $pdo = db();
    $st = $pdo->prepare("UPDATE usuario SET password_hash=? WHERE id_usuario=?");
    $st->execute([$hash, $idUsuario]);
  }

  public function deleteById(int $idUsuario): void {
    $pdo = db();
    $pdo->beginTransaction();
    try {
      if (!$this->findById($idUsuario)) {
        throw new Exception('Docente no encontrado.');
      }
      // Si existen registros relacionados (preguntas, partidas, etc.)
      // deben limpiarse antes o configurarse con ON DELETE CASCADE.
      $st = $pdo->prepare('DELETE FROM docente WHERE id_usuario=?');
      $st->execute([$idUsuario]);
      $st = $pdo->prepare('DELETE FROM usuario WHERE id_usuario=?');
      $st->execute([$idUsuario]);
      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
      throw $e;
    }
  }
}
