<?php
declare(strict_types=1);

class ComputadoraRepository {

  public function listAll(): array {
    $pdo = db();
    $st = $pdo->query("
      SELECT id_computadora, codigo, ip, descripcion, estado, ultimo_ping, created_at
      FROM computadora
      ORDER BY created_at DESC
    ");
    return $st->fetchAll();
  }

  public function create(string $codigo, ?string $ip, ?string $descripcion): int {
    $pdo = db();

    $codigo = strtoupper(trim($codigo));
    $ip = $ip !== null ? trim($ip) : null;
    $descripcion = $descripcion !== null ? trim($descripcion) : null;

    if ($codigo === '') {
      throw new RuntimeException('El código es obligatorio.');
    }

    // Validación simple: evitar códigos duplicados
    $stCheck = $pdo->prepare('SELECT COUNT(*) FROM computadora WHERE codigo = ?');
    $stCheck->execute([$codigo]);
    if ((int)$stCheck->fetchColumn() > 0) {
      throw new RuntimeException('Ya existe una computadora con ese código.');
    }

    $st = $pdo->prepare(
      'INSERT INTO computadora (codigo, ip, descripcion, estado) VALUES (?, ?, ?, "LIBRE")'
    );
    $st->execute([$codigo, $ip ?: null, $descripcion ?: null]);
    return (int)$pdo->lastInsertId();
  }

  public function deleteById(int $id): void {
    $pdo = db();
    $st = $pdo->prepare('DELETE FROM computadora WHERE id_computadora = ?');
    $st->execute([$id]);
  }

  public function findById(int $id): ?array {
    $pdo = db();
    $st = $pdo->prepare('SELECT id_computadora, codigo, ip, descripcion, estado, ultimo_ping, created_at FROM computadora WHERE id_computadora=? LIMIT 1');
    $st->execute([$id]);
    $row = $st->fetch();
    return $row ?: null;
  }

  public function updateById(int $id, string $codigo, ?string $ip, ?string $descripcion): void {
    $pdo = db();
    $codigo = strtoupper(trim($codigo));
    $ip = $ip !== null ? trim($ip) : null;
    $descripcion = $descripcion !== null ? trim($descripcion) : null;

    if ($id <= 0) {
      throw new RuntimeException('Computadora inválida.');
    }
    if ($codigo === '') {
      throw new RuntimeException('El código es obligatorio.');
    }

    // Evitar duplicado de código en otro registro
    $stCheck = $pdo->prepare('SELECT COUNT(*) FROM computadora WHERE codigo = ? AND id_computadora <> ?');
    $stCheck->execute([$codigo, $id]);
    if ((int)$stCheck->fetchColumn() > 0) {
      throw new RuntimeException('Ya existe una computadora con ese código.');
    }

    $st = $pdo->prepare('UPDATE computadora SET codigo=?, ip=?, descripcion=? WHERE id_computadora=?');
    $st->execute([$codigo, $ip ?: null, $descripcion ?: null, $id]);
  }

  public function setEstado(int $id, string $estado): void {
    $pdo = db();
    $st = $pdo->prepare("UPDATE computadora SET estado=? WHERE id_computadora=?");
    $st->execute([strtoupper($estado), $id]);
  }

  public function touchPing(int $id): void {
    $pdo = db();
    $st = $pdo->prepare("UPDATE computadora SET ultimo_ping=NOW() WHERE id_computadora=?");
    $st->execute([$id]);
  }
public function findByIp(string $ip): ?array {
  $pdo = db();
  $st = $pdo->prepare("SELECT * FROM computadora WHERE ip = ? LIMIT 1");
  $st->execute([trim($ip)]);
  $row = $st->fetch();
  return $row ?: null;
}

public function touchOcupadaByIp(string $ip): void {
  $ip = trim($ip);
  if ($ip === '') return;

  $pdo = db();
  $st = $pdo->prepare("
    UPDATE computadora
    SET estado='OCUPADA', ultimo_ping=NOW()
    WHERE ip = ?
  ");
  $st->execute([$ip]);
}

public function touchPingByIp(string $ip): void {
  $ip = trim($ip);
  if ($ip === '') return;

  $pdo = db();
  $st = $pdo->prepare("UPDATE computadora SET ultimo_ping=NOW() WHERE ip = ?");
  $st->execute([$ip]);
}

public function liberarByIp(string $ip): void {
  $ip = trim($ip);
  if ($ip === '') return;

  $pdo = db();
  $st = $pdo->prepare("UPDATE computadora SET estado='LIBRE' WHERE ip = ?");
  $st->execute([$ip]);
}
}
