<?php
declare(strict_types=1);

require_once APP_PATH . "/repositories/ResultadoRepository.php";

class ResultadoController extends Controller
{
  /* =========================================================
     HELPERS
     ========================================================= */

  private function requireLogin(): array {
    Auth::requireLogin();
    $u = Auth::user();
    if (!is_array($u)) {
      http_response_code(401);
      echo "401 - No autenticado";
      exit;
    }
    return $u;
  }

  private function requireEstudiante(): array {
    $u = $this->requireLogin();
    if (($u['rol'] ?? '') !== 'ESTUDIANTE') {
      http_response_code(403);
      echo "403 - Solo estudiantes";
      exit;
    }
    return $u;
  }

  private function requireDocente(): array {
    $u = $this->requireLogin();
    if (($u['rol'] ?? '') !== 'DOCENTE') {
      http_response_code(403);
      echo "403 - Solo docentes";
      exit;
    }
    return $u;
  }

  /* =========================================================
     /resultado
     - ESTUDIANTE: muestra ranking de la partida actual (tu lógica)
     - DOCENTE: muestra panel de puntajes por partida
     ========================================================= */
  public function show(): void
  {
    $u = $this->requireLogin();
    $rol = $u['rol'] ?? '';

    // =========================
    // ESTUDIANTE (igual que tu código)
    // =========================
    if ($rol === 'ESTUDIANTE') {
      Auth::start();
      $quiz = $_SESSION['quiz'] ?? null;

      if (!$quiz) {
        $this->redirect("/SEIR/public/unirse?error=" . urlencode("Primero debes unirte a una partida."));
      }

      $ranking = (new ResultadoRepository())->getRankingPartida((int)$quiz['id_partida']);

      $this->view("resultado/show", [
        "user" => $u,
        "quiz" => $quiz,
        "ranking" => $ranking
      ]);
      return;
    }

    // =========================
    // DOCENTE (nuevo)
    // =========================
    if ($rol === 'DOCENTE') {
      $this->puntajesDocente(); // reutiliza el método de abajo
      return;
    }

    http_response_code(403);
    echo "403 - No autorizado";
  }

  /* =========================================================
     /mis-resultados  (SOLO ESTUDIANTE)
     ========================================================= */
  public function misResultados(): void
  {
    $u = $this->requireEstudiante();

    $rows = (new ResultadoRepository())->listByUsuario((int)$u['id_usuario']);

    $this->view("resultado/mis", [
      "user" => $u,
      "rows" => $rows
    ]);
  }

  /* =========================================================
     PANEL DOCENTE: resumen por partida
     VISTA: app/views/resultado/docente.php  (la creas)
     ========================================================= */
 public function puntajesDocente(): void
{
  $u = $this->requireDocente();

  $pdo = db();

  // En tu BD: partida.id_docente
  $idDocente = (int)($u['id_docente'] ?? ($u['id_usuario'] ?? 0));

  $stmt = $pdo->prepare("
    SELECT
      p.id_partida,
      p.codigo,
      p.titulo,
      p.estado,
      p.created_at,
      COUNT(DISTINCT par.id_participante) AS participantes,
      COUNT(r.id_respuesta) AS respuestas,
      ROUND(100 * SUM(r.puntaje_obtenido) / NULLIF(COUNT(r.id_respuesta),0), 1) AS porcentaje
    FROM partida p
    LEFT JOIN participante par ON par.id_partida = p.id_partida
    LEFT JOIN respuesta r ON r.id_participante = par.id_participante
    WHERE p.id_docente = :id
    GROUP BY p.id_partida, p.codigo, p.titulo, p.estado, p.created_at
    ORDER BY p.created_at DESC
  ");
  $stmt->execute([':id' => $idDocente]);
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  $this->view("resultado/docente", [
    "title"   => "SEIR • Puntajes",
    "pageCss" => "/SEIR/public/assets/css/pages/puntajes.css",
    "user"    => $u,
    "rows"    => $rows
  ]);
}

}
