<?php
declare(strict_types=1);

require_once APP_PATH . "/repositories/QuizRepository.php";
require_once APP_PATH . "/repositories/PartidaRepository.php";
require_once APP_PATH . "/repositories/ParticipanteRepository.php";
require_once APP_PATH . "/repositories/ComputadoraRepository.php";

class QuizController extends Controller {

  private function requireEstudiante(): void {
    Auth::requireLogin();
    $u = Auth::user();
    if (($u['rol'] ?? '') !== 'ESTUDIANTE') {
      http_response_code(403);
      echo "403 - Solo estudiantes";
      exit;
    }
  }

  private function ensureParticipante(int $idPartida): int {
    Auth::start();
    $quizSes = $_SESSION['quiz'] ?? null;
    if (!$quizSes) return 0;

    $idParticipante = (int)($quizSes['id_participante'] ?? 0);
    $partiRepo = new ParticipanteRepository();
    $row = $idParticipante > 0 ? $partiRepo->findById($idParticipante) : null;

    // Si no existe, lo buscamos por (partida, estudiante). Si tampoco existe, lo creamos.
    if (!$row || (int)($row['id_partida'] ?? 0) !== $idPartida) {
      $user = Auth::user();
      $idEst = (int)($user['id_usuario'] ?? 0);
      if ($idEst > 0) {
        $exist = $partiRepo->findByPartidaAndEstudiante($idPartida, $idEst);
        if ($exist) {
          $idParticipante = (int)$exist['id_participante'];
        } else {
          $alias = (string)($quizSes['alias'] ?? ('EST-' . $idEst));
          $idParticipante = $partiRepo->create($idPartida, $idEst, $alias);
        }
        $_SESSION['quiz']['id_participante'] = $idParticipante;
      }
    }

    return $idParticipante;
  }

  private function touchPc(): void {
    Auth::start();
    $quiz = $_SESSION['quiz'] ?? null;
    if (!$quiz) return;
    $ip = $quiz['ip'] ?? ($_SERVER['REMOTE_ADDR'] ?? '');
    if ($ip === '') return;
    try { (new ComputadoraRepository())->touchPingByIp($ip); } catch (Throwable $e) {}
  }

  // Nota: el tiempo restante se calcula usando MySQL (TIMESTAMPDIFF) para evitar
  // desfases de zona horaria entre PHP y MySQL.

  public function show(): void {
    $this->requireEstudiante();
    Auth::start();

    $quizSes = $_SESSION['quiz'] ?? null;
    if (!$quizSes) {
      $this->redirect("/SEIR/public/unirse?error=" . urlencode("Primero debes unirte a una partida."));
    }

    $idPartida = (int)$quizSes['id_partida'];

    // Asegurar que el participante exista (evita FK errors al guardar respuestas)
    $idParticipante = $this->ensureParticipante($idPartida);

    // Asegurar que el participante exista (evita FK errors por imports/reinicios)
    $idParticipante = $this->ensureParticipante($idPartida);
    if ($idParticipante <= 0) {
      $this->redirect('/SEIR/public/unirse?error=' . urlencode('Primero debes unirte a una partida.'));
    }

    // Verificar estado de la partida
    $partRepo = new PartidaRepository();
    $tiempoData = $partRepo->getTiempoData($idPartida);

    if (!$tiempoData || ($tiempoData['estado'] ?? '') !== 'EN_CURSO') {
      $this->redirect("/SEIR/public/sala");
    }

    // Tiempo total (server-side) usando reloj de MySQL
    $segRest = $partRepo->getSegundosRestantes($idPartida);

    // Si acabó el tiempo -> finalizar y mandar a fin
    $partiRepo = new ParticipanteRepository();
    if ($segRest <= 0) {
      try { $partiRepo->setFinalizado($idParticipante); } catch (Throwable $e) {}
      $this->redirect("/SEIR/public/quiz/fin");
    }

    // Si ya envió -> fin
    if ($partiRepo->isFinalizado($idParticipante)) {
      $this->redirect("/SEIR/public/quiz/fin");
    }

    $repo = new QuizRepository();
    $preguntas = $repo->listPreguntas($idPartida);
    $total = count($preguntas);
    if ($total <= 0) {
      $this->redirect("/SEIR/public/sala");
    }

    $orden = (int)($_GET['o'] ?? 1);
    if ($orden < 1) $orden = 1;
    if ($orden > $total) $orden = $total;

    $preg = $repo->getPreguntaByOrden($idPartida, $orden);
    if (!$preg) {
      $this->redirect("/SEIR/public/quiz?o=1");
    }

    $alts = $repo->getAlternativas((int)$preg['id_pregunta']);

    // Mezcla determinística de alternativas por estudiante (anti-copia)
    $seed = crc32($idParticipante . '-' . (int)$preg['id_pregunta']);
    mt_srand($seed);
    shuffle($alts);

    $mapResp = $repo->getRespuestasMap($idPartida, $idParticipante);
    $seleccionada = $mapResp[(int)$preg['id_partida_pregunta']] ?? 0;

    $respondidas = count($mapResp);

    $this->touchPc();

    $this->view("quiz/show", [
      "quiz" => $quizSes,
      "preg" => $preg,
      "alts" => $alts,
      "orden" => $orden,
      "total" => $total,
      "preguntas" => $preguntas,
      "mapResp" => $mapResp,
      "seleccionada" => $seleccionada,
      "respondidas" => $respondidas,
      "segRest" => $segRest,
      "duracionMin" => (int)($tiempoData['duracion_total_min'] ?? 0)
    ]);
  }

  public function responder(): void {
    $this->requireEstudiante();
    Auth::start();
    $quizSes = $_SESSION['quiz'] ?? null;
    if (!$quizSes) {
      $this->redirect("/SEIR/public/unirse?error=" . urlencode("Primero debes unirte a una partida."));
    }

    $idPartida = (int)$quizSes['id_partida'];

    // Asegurar que el participante exista (evita FK errors al guardar respuestas)
    $idParticipante = $this->ensureParticipante($idPartida);

    // Bloqueo si ya finalizó
    $partiRepo = new ParticipanteRepository();
    if ($partiRepo->isFinalizado($idParticipante)) {
      $this->redirect("/SEIR/public/quiz/fin");
    }

    // Tiempo total (server-side)
    $tiempoData = (new PartidaRepository())->getTiempoData($idPartida);
    if (!$tiempoData || ($tiempoData['estado'] ?? '') !== 'EN_CURSO') {
      $this->redirect("/SEIR/public/sala");
    }
    if ((new PartidaRepository())->getSegundosRestantes($idPartida) <= 0) {
      try { $partiRepo->setFinalizado($idParticipante); } catch (Throwable $e) {}
      $this->redirect("/SEIR/public/quiz/fin");
    }

    $idPP = (int)($_POST['id_partida_pregunta'] ?? 0);
    $idAlt = (int)($_POST['id_alternativa'] ?? 0);
    $orden = (int)($_POST['orden'] ?? 1);

    // Guardar respuesta (si eligió)
    if ($idPP > 0 && $idAlt > 0) {
      $t = (float)($_POST['tiempo'] ?? 0);
      (new QuizRepository())->upsertRespuesta($idParticipante, $idPP, $idAlt, $t);
      $this->touchPc();
    }

    $this->redirect("/SEIR/public/quiz?o=" . $orden);
  }

  public function enviar(): void {
    $this->requireEstudiante();
    Auth::start();
    $quizSes = $_SESSION['quiz'] ?? null;
    if (!$quizSes) {
      $this->redirect("/SEIR/public/unirse?error=" . urlencode("Primero debes unirte a una partida."));
    }
    $idParticipante = (int)$quizSes['id_participante'];
    try { (new ParticipanteRepository())->setFinalizado($idParticipante); } catch (Throwable $e) {}
    $this->redirect("/SEIR/public/quiz/fin");
  }

  public function fin(): void {
    $this->requireEstudiante();
    Auth::start();
    $quizSes = $_SESSION['quiz'] ?? null;
    if (!$quizSes) {
      $this->redirect("/SEIR/public/unirse?error=" . urlencode("Primero debes unirte a una partida."));
    }

    $repo = new QuizRepository();
    $idPartida = (int)$quizSes['id_partida'];

    // Asegurar que el participante exista (evita FK errors al guardar respuestas)
    $idParticipante = $this->ensureParticipante($idPartida);

    $respondidas = $repo->countRespondidas($idPartida, $idParticipante);
    $total = $repo->countTotalPreguntas($idPartida);

    $this->view("quiz/fin", [
      "quiz" => $quizSes,
      "respondidas" => $respondidas,
      "total" => $total
    ]);
  }
}
