<?php
declare(strict_types=1);

require_once APP_PATH . "/services/PreguntaService.php";

class PreguntaController extends Controller
{
    private function requireDocenteAdmin(): void
    {
        Auth::requireLogin();
        $u = Auth::user();
        if (!in_array($u['rol'], ['DOCENTE', 'ADMIN'], true)) {
            http_response_code(403);
            echo "403 - No autorizado";
            exit;
        }
    }

    private function preguntasPageMeta(string $title): array
    {
        return [
            "title"   => $title,
            "pageCss" => "/SEIR/public/assets/css/pages/preguntas.css",
        ];
    }

    public function index(): void
    {
        $this->requireDocenteAdmin();
        $service = new PreguntaService();
        $u = Auth::user();

        $this->view("preguntas/index", array_merge(
            $this->preguntasPageMeta("SEIR • Banco de preguntas"),
            [
                "preguntas" => $service->listar((int)$u['id_usuario']),
                "ok"        => $_GET["ok"] ?? "",
                "error"     => $_GET["error"] ?? ""
            ]
        ));
    }

    public function createForm(): void
    {
        $this->requireDocenteAdmin();
        $idCurso = (int)($_GET['curso'] ?? $_GET['id_curso'] ?? 0);


        $this->view("preguntas/create", array_merge(
            $this->preguntasPageMeta("SEIR • Nueva pregunta"),
            [
                "error" => $_GET["error"] ?? "",
                "idCurso" => $idCurso
            ]
        ));
    }

    public function editForm(): void
    {
        $this->requireDocenteAdmin();
        $u  = Auth::user();
        $id = (int)($_GET['id'] ?? 0);

        if ($id <= 0) {
            $this->redirect("/SEIR/public/preguntas?error=" . urlencode("ID inválido"));
        }

        try {
            $service = new PreguntaService();
            $data    = $service->obtener((int)$u['id_usuario'], $id);

            $this->view("preguntas/edit", array_merge(
                $this->preguntasPageMeta("SEIR • Editar pregunta"),
                [
                    "p"     => $data["pregunta"],
                    "alts"  => $data["alternativas"],
                    "error" => $_GET["error"] ?? ""
                ]
            ));
        } catch (Throwable $e) {
            $this->redirect("/SEIR/public/preguntas?error=" . urlencode($e->getMessage()));
        }
    }


    public function destroy(): void
    {
        $this->requireDocenteAdmin();
        $u  = Auth::user();
        $id = (int)($_POST['id_pregunta'] ?? 0);

        try {
            $service = new PreguntaService();
            $service->eliminar((int)$u['id_usuario'], $id);

            $this->redirect("/SEIR/public/preguntas?ok=" . urlencode("Pregunta eliminada."));
        } catch (Throwable $e) {
            $this->redirect("/SEIR/public/preguntas?error=" . urlencode($e->getMessage()));
        }
    }

public function byCurso(): void
{
  $this->requireDocenteAdmin();
  $u = Auth::user();
  $idDocente = (int)($u['id_usuario'] ?? 0);

  $idCurso = (int)($_GET['id'] ?? 0);
  if ($idCurso <= 0) {
    $this->redirect('/SEIR/public/cursos');
    return;
  }

  $service = new PreguntaService();
  $preguntas = $service->listarPorCurso($idDocente, $idCurso);

  $this->view('preguntas/by_curso', [
    'title' => 'SEIR • Preguntas del curso',
    'pageCss' => '/SEIR/public/assets/css/pages/preguntas-curso.css',
    'preguntas' => $preguntas,
    'idCurso' => $idCurso,
    'ok' => $_GET['ok'] ?? '',
    'error' => $_GET['error'] ?? '',
  ]);
}


private function procesarImagen(?array $file): ?string
{
    if (!$file || $file['error'] === UPLOAD_ERR_NO_FILE) {
        return null;
    }

    if ($file['error'] !== UPLOAD_ERR_OK) {
        throw new Exception("Error al subir la imagen.");
    }

    $permitidos = ['image/jpeg','image/png','image/webp'];
    if (!in_array($file['type'], $permitidos, true)) {
        throw new Exception("Formato de imagen no permitido.");
    }

    if ($file['size'] > 2 * 1024 * 1024) {
        throw new Exception("La imagen supera los 2MB.");
    }

    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $nombre = uniqid('preg_', true) . '.' . $ext;

    $destino = APP_PATH . '/../public/assets/uploads/preguntas/';
    if (!is_dir($destino)) {
        mkdir($destino, 0775, true);
    }

    move_uploaded_file($file['tmp_name'], $destino . $nombre);

    return $nombre;
}

public function store(): void
{
    $this->requireDocenteAdmin();
    $u = Auth::user();

    try {
        $imagen = $this->procesarImagen($_FILES['imagen'] ?? null);

        $service = new PreguntaService();
        $service->crear((int)$u['id_usuario'], $_POST, $imagen);

        $this->redirect("/SEIR/public/preguntas?ok=Pregunta creada.");
    } catch (Throwable $e) {
        $this->redirect("/SEIR/public/preguntas/crear?error=" . urlencode($e->getMessage()));
    }
}

public function update(): void
{
    $this->requireDocenteAdmin();
    $u  = Auth::user();
    $id = (int)$_POST['id_pregunta'];

    try {
        $imagen = $this->procesarImagen($_FILES['imagen'] ?? null);

        $service = new PreguntaService();
        $service->actualizar((int)$u['id_usuario'], $id, $_POST, $imagen);

        $this->redirect("/SEIR/public/preguntas?ok=Pregunta actualizada.");
    } catch (Throwable $e) {
        $this->redirect("/SEIR/public/preguntas/editar?id=$id&error=" . urlencode($e->getMessage()));
    }
}


}
