<?php
declare(strict_types=1);

require_once APP_PATH . "/services/PartidaService.php";
require_once APP_PATH . "/services/CursoService.php";

// ✅ Estos 2 los usas dentro de preguntasForm()
require_once APP_PATH . "/repositories/PreguntaRepository.php";
require_once APP_PATH . "/repositories/PartidaRepository.php";

class PartidaController extends Controller {

  private function requireDocenteAdmin(): void {
    Auth::requireLogin();
    $u = Auth::user();
    $rol = strtoupper(trim((string)($u['rol'] ?? '')));
    if (!in_array($rol, ['DOCENTE','ADMIN','PROFESOR','ADMINISTRADOR'], true)) {
      http_response_code(403);
      echo "403 - No autorizado";
      exit;
    }
  }

  private function partidasPageMeta(string $title): array {
    return [
      "title"   => $title,
      "pageCss" => "/SEIR/public/assets/css/pages/partidas.css",
    ];
  }

  // ✅ INDEX con filtros + lista de cursos
  public function index(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();

    $service = new PartidaService();
    $cursoSrv = new CursoService();

    $this->view("partidas/index", array_merge(
      $this->partidasPageMeta("SEIR • Partidas"),
      [
        "partidas" => $service->listarFiltrado((int)$u['id_usuario'], $_GET),
        "cursos"   => $cursoSrv->listar((int)$u['id_usuario']),
        "ok"       => $_GET["ok"] ?? "",
        "error"    => $_GET["error"] ?? ""
      ]
    ));
  }

  // ✅ CREATE FORM con cursos
  public function createForm(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();

    $cursoSrv = new CursoService();

    $this->view("partidas/create", array_merge(
      $this->partidasPageMeta("SEIR • Nueva partida"),
      [
        "error"  => $_GET["error"] ?? "",
        "cursos" => $cursoSrv->listar((int)$u['id_usuario']),
      ]
    ));
  }

  public function store(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();

    try {
      (new PartidaService())->crear((int)$u['id_usuario'], $_POST);
      $this->redirect("/SEIR/public/partidas?ok=" . urlencode("Partida creada. Ahora agrega preguntas."));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas/crear?error=" . urlencode($e->getMessage()));
    }
  }

  // ✅ EDIT FORM (solo si estado=CREADA)
  public function editForm(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_GET['id'] ?? 0);

    if ($id <= 0) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode("ID inválido"));
      return;
    }

    try {
      $service = new PartidaService();
      $cursoSrv = new CursoService();

      $partida = $service->obtener((int)$u['id_usuario'], $id);

      $this->view("partidas/edit", array_merge(
        $this->partidasPageMeta("SEIR • Editar partida"),
        [
          "p"      => $partida,
          "cursos" => $cursoSrv->listar((int)$u['id_usuario']),
          "error"  => $_GET["error"] ?? "",
        ]
      ));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  // ✅ UPDATE (solo si estado=CREADA)
  public function update(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_POST['id_partida'] ?? 0);

    try {
      (new PartidaService())->actualizar((int)$u['id_usuario'], $id, $_POST);
      $this->redirect("/SEIR/public/partidas?ok=" . urlencode("Partida actualizada."));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas/editar?id={$id}&error=" . urlencode($e->getMessage()));
    }
  }

  // ✅ DELETE (solo si estado=CREADA)
  public function destroy(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_POST['id_partida'] ?? 0);

    try {
      (new PartidaService())->eliminar((int)$u['id_usuario'], $id);
      $this->redirect("/SEIR/public/partidas?ok=" . urlencode("Partida eliminada."));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  // ✅ PREGUNTAS (con filtro por curso + grado por defecto)
  public function preguntasForm(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_GET["id"] ?? 0);

    try {
      $service = new PartidaService();

      // 1) obtenemos la partida para saber grado_objetivo y curso
      $tmp = $service->formPreguntas((int)$u['id_usuario'], $id, 'TODOS');
      $partida = $tmp["partida"];

      // ✅ grado default = grado_objetivo (si 1..5). Si 0 => TODOS
      $gradoDefault = ((int)($partida['grado_objetivo'] ?? 0) > 0) ? (string)(int)$partida['grado_objetivo'] : 'TODOS';
      $grado = strtoupper(trim((string)($_GET['grado'] ?? $gradoDefault)));

      // ✅ curso default = id_curso de la partida (si tiene)
      $cursoDefault = (int)($partida['id_curso'] ?? 0);
      $cursoSel = (int)($_GET['curso'] ?? $cursoDefault);

      // ✅ lista cursos para dropdown
      $cursos = (new CursoService())->listar((int)$u['id_usuario']);

      // 2) banco filtrado real por cursoSel si cursoSel>0
      if ($cursoSel > 0) {
        $banco = (new PreguntaRepository())->getBancoByDocenteCursoGrado((int)$u['id_usuario'], $cursoSel, $grado);

        $seleccionadas = (new PartidaRepository())->getPreguntasIds($id);
        $mapOrden = [];
        $mapPuntos = [];
        foreach ($seleccionadas as $s) {
          $idp = (int)$s['id_pregunta'];
          $mapOrden[$idp] = (int)$s['orden'];
          $mapPuntos[$idp] = (int)($s['puntos'] ?? 1);
        }

        $this->view("partidas/preguntas", array_merge(
          $this->partidasPageMeta("SEIR • Partida • Preguntas"),
          [
            "partida"   => $partida,
            "banco"     => $banco,
            "grado"     => $grado,
            "cursoSel"  => $cursoSel,
            "cursos"    => $cursos,
            "mapOrden"  => $mapOrden,
            "mapPuntos" => $mapPuntos,
            "error"     => $_GET["error"] ?? "",
            "ok"        => $_GET["ok"] ?? ""
          ]
        ));
        return;
      }

      // ✅ si cursoSel = 0 => service normal (si la partida tiene curso, igual filtra)
      $data = $service->formPreguntas((int)$u['id_usuario'], $id, $grado);

      $this->view("partidas/preguntas", array_merge(
        $this->partidasPageMeta("SEIR • Partida • Preguntas"),
        [
          "partida"   => $data["partida"],
          "banco"     => $data["banco"],
          "grado"     => $data["grado"],
          "cursoSel"  => $cursoSel,
          "cursos"    => $cursos,
          "mapOrden"  => $data["mapOrden"],
          "mapPuntos" => $data["mapPuntos"],
          "error"     => $_GET["error"] ?? "",
          "ok"        => $_GET["ok"] ?? ""
        ]
      ));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  public function preguntasSave(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_POST["id_partida"] ?? 0);

    try {
      (new PartidaService())->guardarPreguntas((int)$u['id_usuario'], $id, $_POST);
      $this->redirect("/SEIR/public/partidas/preguntas?id=$id&ok=" . urlencode("Preguntas guardadas."));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas/preguntas?id=$id&error=" . urlencode($e->getMessage()));
    }
  }

  public function iniciar(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_POST['id_partida'] ?? 0);

    try {
      (new PartidaService())->iniciar((int)$u['id_usuario'], $id);
      $this->redirect("/SEIR/public/partidas?ok=" . urlencode("Partida iniciada."));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  public function finalizar(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_POST['id_partida'] ?? 0);

    try {
      (new PartidaService())->finalizar((int)$u['id_usuario'], $id);
      $this->redirect("/SEIR/public/partidas?ok=" . urlencode("Partida finalizada."));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  public function ranking(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_GET['id'] ?? 0);

    try {
      $rows = (new PartidaService())->ranking((int)$u['id_usuario'], $id);

      $this->view("partidas/ranking", array_merge(
        $this->partidasPageMeta("SEIR • Ranking"),
        [
          "ranking"    => $rows,
          "id_partida" => $id
        ]
      ));
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  // ✅ CONTROL (ARREGLADO): manda TODO y la vista ya no llama repos
  public function control(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_GET["id"] ?? 0);

    if ($id <= 0) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode("ID inválido"));
      return;
    }

    try {
      $service = new PartidaService();
      $data = $service->controlData((int)$u["id_usuario"], $id);

      $this->view("partidas/control", [
        "title"   => "SEIR • Control de partida",
        "pageCss" => "/SEIR/public/assets/css/pages/partida-control.css",

        "id_partida"   => $id,
        "partida"      => $data["partida"],
        "tiempo"       => $data["tiempo"],
        "resumen"      => $data["resumen"],
        "participantes"=> $data["participantes"],
        "top"          => $data["top"],

        "ok"           => $_GET["ok"] ?? "",
        "error"        => $_GET["error"] ?? ""
      ]);
    } catch (Throwable $e) {
      $this->redirect("/SEIR/public/partidas?error=" . urlencode($e->getMessage()));
    }
  }

  // ✅ endpoint JSON para polling en control
  public function controlData(): void {
    $this->requireDocenteAdmin();
    $u = Auth::user();
    $id = (int)($_GET["id"] ?? 0);

    try {
      $service = new PartidaService();
      $data = $service->controlData((int)$u["id_usuario"], $id);

      header("Content-Type: application/json; charset=utf-8");
      echo json_encode($data);
    } catch (Throwable $e) {
      http_response_code(400);
      header("Content-Type: application/json; charset=utf-8");
      echo json_encode(["error" => $e->getMessage()]);
    }
  }
}
